<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\FeeType;
use App\Models\FeePayment;
use App\Models\LedgerEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FeeController extends Controller
{
    /**
     * Show the form for collecting fee
     */
    public function collect()
    {
        $students = Student::orderBy('name')->get();
        $feeTypes = FeeType::orderBy('name')->get();
        return view('fees.collect', compact('students', 'feeTypes'));
    }

    /**
     * Store fee payment
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'fee_type_id' => 'required|exists:fee_types,id',
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,other',
            'receipt_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        // Check for duplicate receipt number if provided
        if (!empty($validated['receipt_number'])) {
            $existing = FeePayment::where('receipt_number', $validated['receipt_number'])->first();
            if ($existing) {
                return redirect()->route('fees.collect')
                    ->withInput()
                    ->with('error', 'Receipt number already exists. Please use a different receipt number.');
            }
        }

        try {
            DB::transaction(function () use ($validated) {
                $student = Student::findOrFail($validated['student_id']);
                $feeType = FeeType::findOrFail($validated['fee_type_id']);

                // Create fee payment
                $payment = FeePayment::create($validated);

                // Create ledger entry automatically
                LedgerEntry::create([
                    'transaction_type' => 'fee_payment',
                    'reference_type' => FeePayment::class,
                    'reference_id' => $payment->id,
                    'student_id' => $validated['student_id'],
                    'entry_type' => 'income',
                    'amount' => $validated['amount'],
                    'transaction_date' => $validated['payment_date'],
                    'receipt_number' => $validated['receipt_number'] ?? null,
                    'description' => "Fee Payment: {$feeType->name} - Student #{$student->admission_number} ({$student->name})",
                ]);
            });

            return redirect()->route('fees.collect')
                ->with('success', 'Fee collected and ledger updated successfully!');
        } catch (\Exception $e) {
            return redirect()->route('fees.collect')
                ->withInput()
                ->with('error', 'Error processing fee payment: ' . $e->getMessage());
        }
    }

    /**
     * Show fee payment history
     */
    public function history(Request $request)
    {
        $query = FeePayment::with(['student', 'feeType'])->latest('payment_date');

        if ($request->has('student_id') && $request->student_id) {
            $query->where('student_id', $request->student_id);
        }

        if ($request->has('from_date')) {
            $query->where('payment_date', '>=', $request->from_date);
        }

        if ($request->has('to_date')) {
            $query->where('payment_date', '<=', $request->to_date);
        }

        $payments = $query->paginate(20);
        $students = Student::orderBy('name')->get();

        return view('fees.history', compact('payments', 'students'));
    }

    /**
     * Show outstanding fees
     */
    public function outstanding()
    {
        $students = Student::with('feePayments')->get()->map(function ($student) {
            $student->outstanding_amount = $student->arrears;
            return $student;
        })->filter(function ($student) {
            return $student->outstanding_amount > 0;
        })->sortByDesc('outstanding_amount');

        return view('fees.outstanding', compact('students'));
    }
}
