@extends('layouts.app')

@section('title', 'Collect Fee - The Smart School Education')
@section('page-title', 'Collect Fee')

@section('content')
@if(session('error'))
<div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg">
    <div class="flex">
        <div class="flex-shrink-0">
            <svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
            </svg>
        </div>
        <div class="ml-3">
            <p class="text-sm text-red-700">{{ session('error') }}</p>
        </div>
    </div>
</div>
@endif

@if(session('success'))
<div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg">
    <div class="flex">
        <div class="flex-shrink-0">
            <svg class="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
            </svg>
        </div>
        <div class="ml-3">
            <p class="text-sm text-green-700">{{ session('success') }}</p>
        </div>
    </div>
</div>
@endif

<div class="bg-white rounded-lg shadow p-6">
    <h2 class="text-xl font-semibold mb-6">Collect Fee Payment</h2>

    <form action="{{ route('fees.store') }}" method="POST">
        @csrf

        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <!-- Student -->
            <div>
                <label for="student_id" class="block text-sm font-medium text-gray-700 mb-2">Select Student *</label>
                <select name="student_id" id="student_id" required
                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 @error('student_id') border-red-500 @enderror"
                        onchange="updateReceiptNumber()">
                    <option value="">Select Student</option>
                    @foreach($students as $student)
                    <option value="{{ $student->id }}" 
                            data-admission="{{ $student->admission_number }}"
                            {{ old('student_id', request('student_id')) == $student->id ? 'selected' : '' }}>
                        {{ $student->admission_number }} - {{ $student->name }}
                    </option>
                    @endforeach
                </select>
                @error('student_id')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Fee Type -->
            <div>
                <label for="fee_type_id" class="block text-sm font-medium text-gray-700 mb-2">Fee Type *</label>
                <select name="fee_type_id" id="fee_type_id" required
                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 @error('fee_type_id') border-red-500 @enderror"
                        onchange="updateFeeAmount()">
                    <option value="">Select Fee Type</option>
                    @foreach($feeTypes as $feeType)
                    <option value="{{ $feeType->id }}" 
                            data-amount="{{ $feeType->amount }}"
                            {{ old('fee_type_id') == $feeType->id ? 'selected' : '' }}>
                        {{ $feeType->name }} - Rs. {{ number_format($feeType->amount, 2) }}
                    </option>
                    @endforeach
                </select>
                @error('fee_type_id')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Amount -->
            <div>
                <label for="amount" class="block text-sm font-medium text-gray-700 mb-2">Amount (Rs.) *</label>
                <input type="number" name="amount" id="amount" value="{{ old('amount', 0) }}" step="0.01" min="0" required
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 @error('amount') border-red-500 @enderror">
                @error('amount')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Payment Date -->
            <div>
                <label for="payment_date" class="block text-sm font-medium text-gray-700 mb-2">Payment Date *</label>
                <input type="date" name="payment_date" id="payment_date" value="{{ old('payment_date', date('Y-m-d')) }}" required
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Payment Method -->
            <div>
                <label for="payment_method" class="block text-sm font-medium text-gray-700 mb-2">Payment Method *</label>
                <select name="payment_method" id="payment_method" required
                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                    <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>Cash</option>
                    <option value="bank_transfer" {{ old('payment_method') == 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                    <option value="cheque" {{ old('payment_method') == 'cheque' ? 'selected' : '' }}>Cheque</option>
                    <option value="other" {{ old('payment_method') == 'other' ? 'selected' : '' }}>Other</option>
                </select>
            </div>

            <!-- Receipt Number -->
            <div>
                <label for="receipt_number" class="block text-sm font-medium text-gray-700 mb-2">Receipt Number</label>
                <input type="text" name="receipt_number" id="receipt_number" value="{{ old('receipt_number') }}"
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
            </div>
        </div>

        <!-- Notes -->
        <div class="mt-6">
            <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">Notes</label>
            <textarea name="notes" id="notes" rows="3"
                      class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">{{ old('notes') }}</textarea>
        </div>

        <!-- Buttons -->
        <div class="mt-6 flex justify-end gap-4">
            <a href="{{ route('dashboard') }}" class="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                Cancel
            </a>
            <button type="submit" class="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                Collect Fee & Update Ledger
            </button>
        </div>
    </form>
</div>

<script>
// Auto-fill receipt number with admission number when student is selected
function updateReceiptNumber() {
    const select = document.getElementById('student_id');
    const selectedOption = select.options[select.selectedIndex];
    const admissionNumber = selectedOption.getAttribute('data-admission');
    const receiptInput = document.getElementById('receipt_number');
    
    // Only auto-fill if receipt number is empty or if user hasn't manually changed it
    if (admissionNumber && (!receiptInput.value || receiptInput.dataset.autoFilled === 'true')) {
        receiptInput.value = admissionNumber;
        receiptInput.dataset.autoFilled = 'true';
    }
}

// Track manual changes to receipt number
document.addEventListener('DOMContentLoaded', function() {
    const receiptInput = document.getElementById('receipt_number');
    const studentSelect = document.getElementById('student_id');
    
    // If student is pre-selected (from URL), auto-fill receipt number
    if (studentSelect.value) {
        updateReceiptNumber();
    }
    
    // Track if user manually changes receipt number
    receiptInput.addEventListener('input', function() {
        if (this.value !== this.dataset.lastAutoValue) {
            this.dataset.autoFilled = 'false';
        }
    });
    
    // Store the auto-filled value for comparison
    receiptInput.addEventListener('change', function() {
        if (this.dataset.autoFilled === 'true') {
            this.dataset.lastAutoValue = this.value;
        }
    });
});

function updateFeeAmount() {
    const select = document.getElementById('fee_type_id');
    const selectedOption = select.options[select.selectedIndex];
    const amount = selectedOption.getAttribute('data-amount');
    
    if (amount) {
        document.getElementById('amount').value = amount;
    }
}
</script>
@endsection
